module.exports = function (grunt) {
	require('time-grunt')(grunt);
	require('load-grunt-tasks')(grunt, {
		pattern: ['grunt-*', '@*/grunt-*', '!grunt-template-jasmine-requirejs']
	});

	var pkg = grunt.file.readJSON('package.json');
	var localConfig = {
			app: pkg.name,
			dist: 'dist',
			build: 'build',
			testPath: pkg.name + '/test',
			testUrl: 'http://localhost:8080/' + pkg.name+ '/test/',
			dest: process.env.CATALINA_HOME + '/webapps/' + pkg.name,
			loginType: 'nextgen'
		};

	grunt.initConfig({
        eslint: {
            options: {
                configFile: '.eslintrc.js',
            },
            target: ['veteran-appointment-requests/app/**/*.js']
        },
		pkg: pkg,
		config: localConfig,
		clean:{
			dist:['.tmp', '<%= config.dist %>/*']
		},
		replace: {
			login: {
				options: {
					patterns: [
						{
							match: /\{login-url\}/g,
							replacement: '<%= pkg.login[config.loginType] %>'
						},
						{
							match: /\{logout-url\}/g,
							replacement: '<%= pkg.logout[config.loginType] %>'
						}
					]
				},
				files: [
					{
						src:  '<%= config.app %>/_resource-directory.json',
						dest: 'build/_temp-resource-directory.json'
					}
				]
			},
			resource: {
				options: {
					patterns: [
						{
							match: /\{base-url\}/g,
							replacement: '<%= replace.baseUrl %>'
						}
					]
				},
				files: [
					{
						src:  'build/_temp-resource-directory.json',
						dest: '<%= config.app %>/resource-directory.json'
					}
				]
			}
		},
		sass_compile_imports: {
			options:{
				header: [
					'// This file is automatically generated',
					'// by the grunt sass_compile_imports task.',
					'// Do not directly modify this file.',
					''
				]
			},
			va_appointments_app: {
				options: {
					// makes path of imported files relative to target
					replacePath: {
						pattern: '<%= config.app %>/app/',
						replace: '../../'
					},
				},
				target: '<%= config.app %>/app/css/sass/va-appointments-app.scss',
				files: [{
					expand: true,
					cwd: '<%= config.app %>/app/',
					// excludes partial sass files
					src: ['modules/**/*.scss', '!modules/**/_*.scss']
				}]
			}
		},
		compass:{
			options: {
				sassDir: '<%= config.app %>/app/css/sass',
				cssDir: '<%= config.app %>/app/css',
				//imagesDir: '<%= yeoman.app %>/_assets/images',
				require: 'breakpoint',
				relativeAssets: false,
				assetCacheBuster: false,
				noLineComments: true,
				debugInfo: false,
				raw: 'Sass::Script::Number.precision = 10\n'
			},
			server: {
				options: {
					debugInfo: false
				}
			}
		},
		jasmine: {
			app: {
				options: {
					keepRunner: true,
					outfile: "<%= config.testPath %>/jasmine_runner/_SpecRunner.html",
					specs: ["<%= config.app %>/app/**/*_spec.js"],
					template: require('grunt-template-jasmine-requirejs'),
					templateOptions:{
						requireConfigFile: ['<%= config.app %>/app/config.js','<%= config.testPath %>/specs/testConfig.js']
					}
				}
			}
		},
		jshint: {
			options: {
				jshintrc: '.jshintrc',
				reporter: require('jshint-stylish')
			},
			all: [
				'Gruntfile.js',
				'<%= config.app %>/app/js/**/*.js'
			]
		},
		karma: {
			unit: {
				configFile: 'karma.conf.js'
			}
		},
		uglify: {
			options: {
				mangle: {
					reserved: ['$', '_', 'Modernizr']
				}
			}
		},
		requirejs: { // base task that optimize (multitask) will use
			dist: {
				options: {
					baseUrl: '<%= config.app %>/app',
					optimize: 'none',
					paths: {
						'externalLinksJson': 'empty:',
						'resourceDirectoryJson': 'empty:',
						'notificationInfoJson': 'empty:',
						'featureFlagJson': 'empty:'
					},
					preserveLicenseComments: false,
					useStrict: true,
					wrapShim: true,
					wrap: true,
					findNestedDependencies: true
				}
			}
		},
		useminPrepare: {
			options: {
				dest: '<%= config.dist %>'
			},
			html: '<%= config.app %>/index.html'
		},
		usemin: {
			options: {
				dirs: ['<%= config.dist %>']
			},
			html: ['<%= config.dist %>/{,*/}*.html'],
			css: ['<%= config.dist %>/app/css/{,*/}*.css']
		},
		imagemin: {
			dist: {
				files: [{
					expand: true,
					cwd: '<%= config.app %>/app/images',
					src: [
						'{,*/}*.{png,jpg,jpeg}',
						'**/*.{png,jpg,jpeg}'
					],
					dest: '<%= config.dist %>/app/images'
				}]
			}
		},
		htmlmin: {
			dist: {
				files: [{
					expand: true,
					cwd: '<%= config.app %>',
					src: '*.html',
					dest: '<%= config.dist %>'
				}]
			}
		},
		copy: {
			dist: {
				files: [{
					expand: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: [
						'*.{ico,txt,png,json,pdf}',
						'!_*resource-directory.json',
						'resource-directory.json',
						'config-template.json',
						'app/css/fonts/{,*/}*.*',
						'app/images/*.{gif,png}',
						'app/images/**/*.{png,jpg,jpeg}',
						'app/**/*.pdf',
						'app/**/notification-info-popup-content.json'
					],
					dest: '<%= config.dist %>'
				},{
					expand: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: 'libs/css/datepicker/*.css',
					dest: '<%= config.dist %>/app/css/datepicker'
				},{
					expand: true,
					flatten: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: 'libs/css/images/*.{gif,png}',
					dest: '<%= config.dist %>/app/images'
				}]
			},
			dev: {
				files: [{
					expand: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: ['**', '!_*resource-directory.json', '!**/*_spec.js'],
					dest: '<%= config.dist %>'
				}]
			},
			toBuild:{
				files: [{
                    expand: true,
                    dot: true,
                    cwd: '<%= config.app %>',
					src: ['**', '!_*resource-directory.json', '!**/*_spec.js', '!test/**', '!WEB-INF/**'],
					dest: '<%= config.dist %>/'
				},{
					expand: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: 'libs/css/datepicker/*.css',
					dest: '<%= config.dist %>/app/css/datepicker'
				},{
					expand: true,
					flatten: true,
					dot: true,
					cwd: '<%= config.app %>',
					src: 'libs/css/images/*.{gif,png}',
					dest: '<%= config.dist %>/app/images'
				}]
			}
		},
		rev: {
			dist: {
				files: {
					src: [
						'<%= config.dist %>/app/js/{,*/}*.js',
						'<%= config.dist %>/app/css/{,*/}*.css'
					]
				}
			}
		},
		watch: {
			app: {
				files: ['<%= config.app %>/app/**'],
				tasks: ['newer:copy:dev']
			}
		},
		connect: {
			dist: {
				options: {
	        		port: 9001,
	        		base: '.',
	        		keepaliave: true,
	        		open: 'http://anywhere:9001/dist'
      			}
			}
		},
		shell: {
			testApp: {
				command: 'node_modules/phantomjs/bin/phantomjs <%= config.app %>test/runner.js <%= config.testUrl  %> 1200',
				options: {
					stdout: true
				}
			},
			version: {
				command: 'git log -1 --pretty=format:%h',
				options: {
					callback: function log(err, stdout, stderr, cb) {
						console.dir(stdout);
						grunt.file.write('veteran-appointment-requests/version.json', JSON.stringify({
							title: pkg.title,
							version: pkg.version,
							metaRevision: stdout,
							date: grunt.template.today()
						}));
						cb();
					}
				}
			},
			revision: {
				command: 'git rev-list HEAD --count',
				options: {
					callback: function log(err, stdout, stderr, cb) {
						console.dir(stdout);
						grunt.file.write('veteran-appointment-requests/version.json', JSON.stringify({
							title: pkg.title,
							version: pkg.version,
							build: stdout.replace('\n', ''),
							metaRevision: grunt.file.readJSON('veteran-appointment-requests/version.json')['metaRevision'],
							date: grunt.template.today()
						}));
						cb();
					}
				}
			}
		}
	});

	grunt.registerTask('replace-resource', "replace resource's baseUrl", function() {
    	grunt.config.set('replace.baseUrl', (grunt.option('baseUrl') || "../..") );
    	grunt.task.run('replace:login');
    	grunt.task.run('replace:resource');
    });

	grunt.registerTask('compile_sass', ['sass_compile_imports', 'compass']);
	grunt.registerTask('test', ['eslint', 'karma']);
	grunt.registerTask('default', ['clean:build']);
	grunt.registerTask('version', ['shell:version']);

	grunt.registerTask('build', [
		'clean:dist',
		'shell:version',
		'compile_sass',
		'replace-resource',
		'useminPrepare',
		'requirejs',
		'imagemin',
		'htmlmin',
		'concat',
		'cssmin',
		'uglify',
		'copy:dist',
		'rev',
		'usemin'
	]);

	grunt.registerTask('default', [
		'jshint',
		'test',
		'build'
	]);
	grunt.registerTask('build:dev', [
		'compile_sass',
		'clean:dist',
		'shell:version',
		'replace-resource',
		'copy:toBuild'
	]);

	grunt.registerTask('serve', [
		'build:dev',
		'connect:dist',
		'watch'
	]);

};

